/************************************************************************
 * NAME:	rgt
 *
 * DESCR:	Resered Group Table implementation.  This implements
 *		the reading and writing of the RGT.  It is normally
 *		cached and writen when necessary.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
#include "hdos.h"

/************************************************************************
 * NAME:	hdos_rgt_read() & hdos_rgt_write()
 *
 * DESCR:	Reads and writes the RGT.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	- requires that the label has already been read
 ************************************************************************/
int
hdos_rgt_read(struct hdosfs *hdosfs)
{
    if (!HDOS_NORGT(hdosfs)) {
	return(hdos_getlsect(hdosfs,hdosfs->labelbuf.rgtsec,hdosfs->RGT));
    } else {
	return(TRUE);
    }
}

int
hdos_rgt_write(struct hdosfs *hdosfs)
{
    if (!HDOS_NORGT(hdosfs)) {
	return(hdos_putlsect(hdosfs,hdosfs->labelbuf.rgtsec,hdosfs->RGT));
    } else {
	return(TRUE);
    }
}

/************************************************************************
 * NAME:	hdos_rgt_compose()
 *
 * DESCR:	Given that everything has been loaded, compose a new
 *		RGT, checking that the current disk looks OK.  The old
 *		RGT is consulted if it exists.
 *
 * ARGS:	
 *
 * RETURNS:	the given (at least) 200 place array is filled in with
 *		the composed RGT.
 *
 * NOTES:	- If not RGT is present, the first track is considered
 *		  reserved.
 *		- groups 0 and 1 are always considered reserved.
 *		- if an RGT is present, all 0xff's are marked as reserved.
 *		  (which is in addition to sectors 0 and 1.
 ************************************************************************/
unsigned char *
hdos_rgt_compose(struct hdosfs *hdosfs, unsigned char *cRGT)
{
    struct hdos_dir	*hdir = &hdosfs->dirbuf;
    int			 i,j;

    for (i=HDOS_GROUPS(hdosfs); i--; ) {	/* zero out the new RGT	*/
	cRGT[i] = 0;
    }

    if (HDOS_NORGT(hdosfs)) {	/* reserve the first track	*/
	for (i=HDOS_SECTORS(hdosfs)/HDOS_GROUPSIZE(hdosfs); i--; ) {
	    cRGT[i] = 0xff;
	}
    } else {			/* just copy the existing RGT	*/
	for (i=HDOS_GROUPS(hdosfs); i--; ) {
	    if (hdosfs->RGT[i] == 0xff)
		cRGT[i] = hdosfs->RGT[i];
	}
    }

    cRGT[0] = 0;		/* first two groups reserved	*/
    cRGT[1] = 0;

    /* now compose the rest of the RGT appropriately		*/

    for (i=0; i < hdir->blocks; i++) {
	for (j=0; j < hdir->block[i].entries; j++) {
	    struct hdos_dirent *entry = &hdir->block[i].entry[j];

	    if (HDOS_ENTRY_UNUSED_LAST(entry)) {
		break;
	    }
	    if (!HDOS_ENTRY_UNUSED(entry)) {
		int	size = 0;
		int	g;

		for ( g=entry->firstgroup; g!=0; g = hdosfs->GRT[g] ) {
		    cRGT[g] += 1;
		    if (g==entry->lastgroup) {
			if (hdosfs->GRT[g] != 0) {
			    M2("Warning on %8s.%3s, bad termination on group chain\n",
			       entry->name, entry->ext);
			}
			break;
		    }
		}
	    }
	}
    }

    return(cRGT);
}

/************************************************************************
 * NAME:	hdos_rgt_ok()
 *
 * DESCR:	Returns TRUE if the given RGT looks OK, FALSE otherwise.
 *
 * ARGS:
 *		rgt is the composed rgt to be checked.
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
int
hdos_rgt_ok(struct hdosfs *hdosfs, unsigned char *rgt)
{
    int			 i;

    for (i=HDOS_GROUPS(hdosfs); i--; ) {

	/* it's really good enough just to check for 00, 01, ff	*/

	if (rgt[i] != 0 && rgt[i] != 1 && rgt[i] != 0xff) {
	    return(FALSE);
	}
    }
    return(TRUE);
}

/************************************************************************
 * NAME:	hdos_rgt_install()
 *
 * DESCR:	Given a new (probably composed) RGT, install it as the
 *		current RGT.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
void
hdos_rgt_install(struct hdosfs *hdosfs, unsigned char *RGT)
{
    int i;

    for (i=HDOS_GROUPS(hdosfs); i--; ) {
	hdosfs->RGT[i] = RGT[i];
    }
}

hdos_rgt_report(unsigned char *rgt)
{
    int	i,j;

    /* the RGT is 200 groups so it's divided into 20 by 10 for this report */

    M("      0   1   2   3   4   5   6   7   8   9\n");
    for (i=0; i < 20; i++) {
	M1("%2d:",i);
	for (j=0; j < 10; j++) {
	    M1(" %3d",rgt[i*10+j]);
	}
	M("\n");
    }
}
